/**
* \file: WaylandContext.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <cstring>
#include <poll.h>
#include <sys/prctl.h>
#include <sys/eventfd.h>
#include <adit_logging.h>
#include "WaylandContext.h"

LOG_IMPORT_CONTEXT(aauto_input)

using namespace std;

namespace adit { namespace aauto
{

WaylandContext::DeviceListener::~DeviceListener()
{
}

WaylandContext::WaylandContext(IAditInputSourceCallbacks* inCallbacks)
{
    compositor = nullptr;
    shm = nullptr;
    registry = nullptr;

    display = nullptr;
    shareWlDisplay = false;
    inputQueue = nullptr;
    adapterContext = nullptr;
    shmFormat = 0;

    touchListener = nullptr;

    pointerListener = nullptr;

    threadRunning = false;
    threadId = 0;
    processIntervalUSec = 0;
    shutDownEventFd = -1;

    mCallbacks = inCallbacks;

    staticInitialize();
}
WaylandContext::WaylandContext(struct wl_display* inDisplay,
        IAditInputSourceCallbacks* inCallbacks)
{
    compositor = nullptr;
    shm = nullptr;
    registry = nullptr;

    display = inDisplay;
    shareWlDisplay = true;
    inputQueue = nullptr;
    adapterContext = nullptr;
    shmFormat = 0;

    touchListener = nullptr;

    pointerListener = nullptr;

    threadRunning = false;
    threadId = 0;
    processIntervalUSec = 0;
    shutDownEventFd = -1;

    mCallbacks = inCallbacks;

    staticInitialize();
}

WaylandContext::~WaylandContext()
{
    LOGD_DEBUG((aauto_input, "destroy Wayland context"));

    StopInputThread();

    std::list<WlSeat*>::iterator it;

    {
        /* call Autolock in a scope */
        Autolock l(&mLockSeatList);
        /* destroy all WlSeat objects */
        for(it = mSeatList.begin(); it != mSeatList.end(); it++)
        {
            LOGD_DEBUG((aauto_input, "Destroy WlSeat object for seat with name=%s", (*it)->GetSeatName()));
            /* calls destructor of class WlSeat which destroys wl_* objects */
            delete (*it);
        }
        mSeatList.clear();
    }

    aauto_safe_call(wl_event_queue_destroy, inputQueue);
    aauto_safe_call(wl_compositor_destroy, compositor);
    aauto_safe_call(wl_registry_destroy, registry);
    aauto_safe_call(wl_shm_destroy, shm);

    if (true != shareWlDisplay)
    {
        if (adapterContext)
        {
            compositor_shim_terminate(adapterContext);
        }

        aauto_safe_call(wl_display_disconnect, display);
    }
}

bool WaylandContext::Initialize()
{
    int wlError = -1;

    /*
     * Connect to a Wayland display
     * If name is NULL, its value will bee replaced with the
     * WAYLAND_DISPLAY environment variable if it is set,
     * otherwise display "wayland-0" will be used.
     */
    if (true != shareWlDisplay)
    {
        display = wl_display_connect(nullptr);
        if (display == nullptr)
        {
            LOG_ERROR((aauto_input, "wl_display_connect failed"));
            return false;
        }
    }

    registry = wl_display_get_registry(display);
    if (registry == nullptr)
    {
        LOG_ERROR((aauto_input, "wl_display_get_registry failed"));
        return false;
    }

    wlError = wl_registry_add_listener(registry, &registryListener, this);
    if (wlError < 0)
    {
        LOG_ERROR((aauto_input, "wl_registry_add_listener failed %d", wlError));
        return false;
    }

    inputQueue = wl_display_create_queue(display);
    if (inputQueue == nullptr)
    {
        LOG_ERROR((aauto_input, "failed to create a queue for input events"));
        return false;
    } else {
        // Registry events will come to inputQueue
        wl_proxy_set_queue((struct wl_proxy*) registry, inputQueue);
    }

    // wait for Registry Global event
    wlError = wl_display_roundtrip_queue(display, inputQueue);
    if (wlError < 0)
    {
        LOG_ERROR((aauto_input, "Wayland round trip to get Registry Global event failed %d", wlError));
        return false;
    }

    // wait for Seat Capabilities event, Shm Format event also arrives
    wlError = wl_display_roundtrip_queue(display, inputQueue);
    if (wlError < 0)
    {
        LOG_ERROR((aauto_input, "Wayland round trip to get Seat Capabilities event failed %d", wlError));
        return false;
    }

    if (true != shareWlDisplay)
    {
        adapterContext = compositor_shim_initialize(display);
        if (!adapterContext)
        {
            LOG_ERROR((aauto_input, "compositor_shim_initialize failed"));
            return false;
        }
    }

    LOGD_DEBUG((aauto_input, "WaylandContext initialization is success"));

    return true;
}

void WaylandContext::SetTouchListener(struct wl_touch_listener* inListener,
        shared_ptr<DeviceListener> inObj)
{
     touchListener = inListener;
     touchListenerObj = inObj;
}

void WaylandContext::SetPointerListener(struct wl_pointer_listener* inListener,
        shared_ptr<DeviceListener> inObj)
{

    pointerListener = inListener;
    pointerListenerObj = inObj;
}

bool WaylandContext::StartInputThread(uint32_t inProcessIntervalUSec)
{
/* PRQA: Lint Message 61: boolean compare */
/*lint -save -e61*/
    if (threadRunning)
/*lint -restore*/
    {
        LOG_ERROR((aauto_input, "WaylandContext input thread is already running"));
        return false;
    }

    /* create eventFd to send stop event */
    if (0 > (shutDownEventFd = eventfd(0, 0))) {
        LOG_ERROR((aauto_input, "Failed to create shutdown eventFd, error=%d", errno));
        return false;
    }

    threadRunning = true;
    processIntervalUSec = inProcessIntervalUSec;

    int ret = pthread_create(&threadId, NULL, inputThread, this);
    if (ret != 0)
    {
        threadId = 0;
        threadRunning = false;
        LOG_ERROR((aauto_input, "WaylandContext pthread_create() failed: %d,%d,%s", ret, errno, strerror(errno)));
        return false;
    }

    return true;
}

void WaylandContext::StopInputThread()
{
/* PRQA: Lint Message 61: boolean compare */
/*lint -save -e61*/
    if (threadRunning)
/*lint -restore*/
    {
        threadRunning = false;

        if (shutDownEventFd >= 0) {
            // trigger shutdown event
            if (eventfd_write(shutDownEventFd, shutDownEvent) != 0){
                LOG_ERROR((aauto_input, "Failed to trigger input thread shutdown event"));
            }
        } else {
            LOG_ERROR((aauto_input, "shutDownEventFd is invalid"));
        }

        if (threadId > 0)
        {
            int ret = pthread_join(threadId, nullptr);
            if (ret != 0)
            {
                // if pthread_join fails just log the error using errno
                LOG_ERROR((aauto_input, "pthread_join failed with error %d %s", errno, strerror(errno)));
            }
            else
            {
                LOGD_DEBUG((aauto_input, "WaylandContext::StopInputThread()  inputThread joined"));
                threadId = 0;
            }
        }
        else
        {
            LOG_WARN((aauto_input, "WaylandContext::StopInputThread()  threadId is invalid"));
        }

        if (shutDownEventFd >= 0) {
            close(shutDownEventFd);
            shutDownEventFd = -1;
        }
    }
}

// ========== private methods ==========

struct wl_registry_listener WaylandContext::registryListener;
struct wl_shm_listener WaylandContext::shmListener;
struct wl_seat_listener WaylandContext::seatListener;
bool WaylandContext::staticInitialized = false;

void* WaylandContext::inputThread(void* inData)
{
    WaylandContext* me = static_cast<WaylandContext*>(inData);
    aauto_return_value_on_invalid_argument(aauto, me == nullptr, nullptr);

    // set thread name
    prctl(PR_SET_NAME, "WaylandTouch", 0, 0, 0);

    LOGD_DEBUG((aauto_input, "input poll thread started"));

/* PRQA: Lint Message 61: boolean compare */
/*lint -save -e61*/
    while (me->threadRunning)
/*lint -restore*/
    {
        int wlError = -1;

        while (wl_display_prepare_read_queue(me->display, me->inputQueue) != 0)
        {
            // dispatch events which might be read already
            wlError = wl_display_dispatch_queue_pending(me->display, me->inputQueue);
            if (wlError < 0)
            {
                LOG_ERROR((aauto_input, "Dispatching events, which are already read, failed "
                        "errno: %d - Exit input thread", errno));
                return nullptr;
            }
        }

        wlError = wl_display_flush(me->display);
        if (wlError < 0)
        {
            LOG_ERROR((aauto_input, "Wayland flush failed errno: %d - Exit input thread",
                    errno));
            wl_display_cancel_read(me->display);
            return nullptr;
        }

        // unfortunately we have to poll for input events
        struct pollfd pfd[2];
        // shutdown events
        pfd[0].fd = me->shutDownEventFd;
        pfd[0].events = me->shutDownEvent;
        // input events
        pfd[1].fd = wl_display_get_fd(me->display);
        pfd[1].events = POLLIN;

        int retval = poll(pfd, 2, -1);
        if(retval <= 0)
        {
            wl_display_cancel_read(me->display);

        // handle shut down event before Wayland event
        } else if (pfd[0].revents & me->shutDownEvent) {

            wl_display_cancel_read(me->display);
            break;

        } else if (pfd[1].revents & POLLIN) {

            wlError = wl_display_read_events(me->display);
            if (wlError < 0)
            {
                LOG_ERROR((aauto_input, "Reading wayland event failed errno: %d", errno));
                if (me->mCallbacks != nullptr)
                {
                    me->mCallbacks->notifyErrorCallback(INPUT_EVENT_DISPATCH_ERROR);
                }
            }

            wlError = wl_display_dispatch_queue_pending(me->display, me->inputQueue);
            if (wlError < 0)
            {
                LOG_ERROR((aauto_input, "Dispatching the input queue failed errno: %d", errno));
                if (me->mCallbacks != nullptr)
                {
                    me->mCallbacks->notifyErrorCallback(INPUT_EVENT_DISPATCH_ERROR);
                }
            }
        }
    }

    LOGD_DEBUG((aauto_input, "input poll thread ended"));
    return nullptr;
}

void WaylandContext::staticInitialize()
{
    if (!staticInitialized)
    {
        memset(&seatListener, 0, sizeof(struct wl_seat_listener));
        seatListener.capabilities = onSeatCapabilities;
        seatListener.name = onSeatHandleGetName;

        memset(&shmListener, 0, sizeof(wl_shm_listener));
        shmListener.format = onShmFormats;

        memset(&registryListener, 0, sizeof(wl_registry_listener));
        registryListener.global = onRegistryGlobal;
        registryListener.global_remove = onRegistryGlobalRemove;

        staticInitialized = true;
    }
}

void WaylandContext::onShmFormats(void* inMe, struct wl_shm* inShm,
        uint32_t inFormat)
{
    (void)inShm;
    aauto_return_on_invalid_argument(aauto, inMe == nullptr);

    auto me = static_cast<WaylandContext*>(inMe);

    if (inFormat == WL_SHM_FORMAT_C8)
    {
        me->shmFormat = WL_SHM_FORMAT_C8;
    }
}

void WaylandContext::onRegistryGlobal(void* inMe, struct wl_registry* inRegistry, uint32_t inName,
        const char* inInterface, uint32_t inVersion)
{
    (void)inVersion;
    aauto_return_on_invalid_argument(aauto, inMe == nullptr);
    aauto_return_on_invalid_argument(aauto, inRegistry == nullptr);

    auto me = static_cast<WaylandContext*>(inMe);

    if (true != me->shareWlDisplay)
    {
        if (!strcmp(inInterface, "wl_compositor"))
        {
            me->compositor = static_cast<struct wl_compositor*>(
                    wl_registry_bind(inRegistry, inName, &wl_compositor_interface, 1));
            if (me->compositor == nullptr)
            {
                LOG_ERROR((aauto_input, "wl_registry_bind failed for wl_registry_bind"));
            }
        }

        if (!strcmp(inInterface, "wl_shm"))
        {
            me->shm = static_cast<struct wl_shm*>(
                    wl_registry_bind(inRegistry, inName, &wl_shm_interface, 1));
            if (me->shm != nullptr)
            {
                int error = wl_shm_add_listener(me->shm, &me->shmListener, me);
                if (error < 0)
                {
                    LOG_ERROR((aauto_input, "wl_shm_add_listener failed: %d", error));
                }
            }
            else
            {
                LOG_ERROR((aauto_input, "wl_registry_bind failed for wl_shm"));
            }
        }
    }

    if (!strcmp(inInterface, "wl_seat"))
    {
        WlSeat *pWlSeat = new WlSeat();
        if (pWlSeat != nullptr)
        {
            struct wl_seat* wlSeat = static_cast<struct wl_seat*>(wl_registry_bind(
                                                                  inRegistry, inName, &wl_seat_interface, 3));
            if (wlSeat != nullptr)
            {
                int error = wl_seat_add_listener(wlSeat, &me->seatListener, pWlSeat);
                if (error < 0)
                {
                    LOG_ERROR((aauto_input, "wl_seat_add_listener failed: %d", error));
                }
                /* Seat events will come to inputQueue */
                wl_proxy_set_queue((struct wl_proxy*) wlSeat, me->inputQueue);

                pWlSeat->SetWLSeat(wlSeat);
                pWlSeat->SetParentWaylandContext(me);
                pWlSeat->SetSeatId(inName);
                /* add WlSeat object to list */
                me->AddWlSeat(pWlSeat);
            }
            else
            {
                LOG_ERROR((aauto_input, "wl_registry_bind failed for wl_seat"));
            }
        }
        else
        {
            LOG_ERROR((aauto_input, "Unable to allocate the WlSeat object"));
        }
    }
}

void WaylandContext::onRegistryGlobalRemove(void* inMe, struct wl_registry* inRegistry,
        uint32_t inName)
{
    (void)inRegistry;
    aauto_return_on_invalid_argument(aauto, inMe == nullptr);

    auto me = static_cast<WaylandContext*>(inMe);

    std::list<WlSeat*>::iterator it;

    Autolock l(&me->mLockSeatList);
    std::list<WlSeat*> seatList = me->getWlSeatList();

    /* destroy all WlSeat objects */
    for(it = seatList.begin(); it != seatList.end(); it++)
    {
        if ((*it)->GetSeatId() == inName )
        {
            LOGD_DEBUG((aauto_input, "Destroy WlSeat object for seat with name=%s", (*it)->GetSeatName()));
            /* calls destructor of class WlSeat which destroys wl_* objects */
            delete (*it);
            seatList.erase(it);
            break;
        }
    }
}

void WaylandContext::onSeatHandleGetName(void* inData, struct wl_seat* inSeat, const char *name)
{
    aauto_return_on_invalid_argument(aauto, inData == nullptr);
    aauto_return_on_invalid_argument(aauto, inSeat == nullptr);
    aauto_return_on_invalid_argument(aauto, name == nullptr);

    auto pWlSeat = static_cast<WlSeat*> (inData);
    pWlSeat->SetSeatName(name);
    LOG_INFO((aauto_input, "%s()  Got seat with name=%s", __FUNCTION__, name));
}

void WaylandContext::onSeatCapabilities(void* inData, struct wl_seat* inSeat, uint32_t inCaps)
{
    aauto_return_on_invalid_argument(aauto, inData == nullptr);
    aauto_return_on_invalid_argument(aauto, inSeat == nullptr);

    auto pWlSeat = static_cast<WlSeat*> (inData);
    auto me = pWlSeat->GetParentWaylandContext();

    struct wl_seat* wlSeat = pWlSeat->GetWLSeat();
    if (wlSeat == nullptr) {
        LOG_ERROR((aauto_input, "wl_seat was not created"));
        return;
    }

    /* if touch listener is set */
    if (me->touchListener != nullptr && (inCaps & WL_SEAT_CAPABILITY_TOUCH))
    {
        LOGD_DEBUG((aauto_input, "register touch listener"));

        struct wl_touch *wlTouch = pWlSeat->GetWLTouch();
        if (!wlTouch)
        {
            wlTouch = wl_seat_get_touch(wlSeat);
            if (wlTouch != nullptr)
            {
                wl_touch_set_user_data(wlTouch, (void *)(me->touchListenerObj.get()));
                int wlError = wl_touch_add_listener(wlTouch, me->touchListener,
                                                    (void *)me->touchListenerObj.get());
                if (wlError < 0)
                {
                    aauto_safe_call(wl_touch_release, wlTouch);
                    LOG_ERROR((aauto_input, "wl_touch_add_listener failed"));
                }
            }
            else
            {
                LOG_ERROR((aauto_input, "wl_seat_get_touch failed"));
            }
        }
        else
        {
            LOG_WARN((aauto_input, "wl_touch already exists for this seat"));
        }

        pWlSeat->SetWLTouch(wlTouch);
    }
    else
    {
        if (me->touchListener != nullptr)
            LOGD_DEBUG((aauto_input, "no touch listener to register"));
        else
            LOGD_DEBUG((aauto_input, "screen caps don't support touch listener"));
    }

    /* if pointer listener is set */
    if (me->pointerListener != nullptr && (inCaps & WL_SEAT_CAPABILITY_POINTER))
    {
        LOGD_DEBUG((aauto_input, "register pointer listener"));

        struct wl_pointer *wlPointer = pWlSeat->GetWLPointer();
        if (!wlPointer)
        {
            wlPointer = wl_seat_get_pointer(wlSeat);
            if (wlPointer != nullptr)
            {
                wl_pointer_set_user_data(wlPointer, (void *)(me->pointerListenerObj.get()));
                int wlError = wl_pointer_add_listener(wlPointer, me->pointerListener,
                                                      (void *)(me->pointerListenerObj.get()));
                if (wlError < 0)
                {
                    aauto_safe_call(wl_pointer_release, wlPointer);
                    LOG_ERROR((aauto_input, "wl_pointer_add_listener failed"));
                }
            }
            else
            {
                LOG_ERROR((aauto_input, "wl_seat_get_pointer failed"));
            }
        }
        else
        {
            LOG_WARN((aauto_input, "wl_pointer already exists for this seat"));
        }

        pWlSeat->SetWLPointer(wlPointer);
    }
    else
    {
        if (me->pointerListener != nullptr)
            LOGD_DEBUG((aauto_input, "no pointer listener to register"));
        else
            LOGD_DEBUG((aauto_input, "screen caps don't support pointer listener"));
    }


    /**
     * Same stuff as above should be done for keyboardListener
     *   if (me->keyboardListener != nullptr && (inCaps & WL_SEAT_CAPABILITY_KEYBOARD))
     *   {
     *   }
     */
}

} } // namespace adit { namespace aauto
